<?php
/**
 * POST /api/report.php
 * Public endpoint — users report abusive sites.
 * Rate limited (5 reports per hour per IP).
 */

require_once __DIR__ . '/../includes/security.php';

Security::requirePost();

// Fix #9: CSRF check
$reportInput = json_decode(file_get_contents('php://input'), true) ?: $_POST;
$postedCsrf = $reportInput['_csrf'] ?? '';
$cookieCsrf = $_COOKIE['_csrf'] ?? '';
if ($cookieCsrf === '' || !hash_equals($cookieCsrf, $postedCsrf)) {
    Security::json(['error' => 'Invalid request token. Refresh and try again.'], 403);
}

$cfg = require __DIR__ . '/../includes/config.php';
$ip = Security::clientIp();

// Stricter rate limit for reports (5/hour)
if (!Security::checkRate($ip . ':report', 5, 3600)) {
    Security::json(['error' => 'Too many reports. Try again later.'], 429);
}

$input = $reportInput;

$address = trim($input['address'] ?? '');
$reason  = $input['reason'] ?? 'other';
$detail  = trim($input['detail'] ?? '');

// Validate
if (!preg_match('/^[a-z2-7]{56}\.onion$/', $address)) {
    Security::json(['error' => 'Invalid onion address'], 400);
}

$validReasons = ['csam', 'scam', 'malware', 'miscat', 'other'];
if (!in_array($reason, $validReasons)) {
    $reason = 'other';
}

$detail = mb_substr($detail, 0, 500);
$ipHash = hash('sha256', $ip);

$db = DB::get();

// Check for duplicate pending report from same IP
$dup = $db->prepare(
    "SELECT 1 FROM reports WHERE address = ? AND ip_hash = ? AND status = 'pending'"
);
$dup->execute([$address, $ipHash]);
if ($dup->fetch()) {
    Security::json(['status' => 'already_reported']);
}

// Cap total pending reports per address (prevent flood)
$cnt = $db->prepare("SELECT COUNT(*) FROM reports WHERE address = ? AND status = 'pending'");
$cnt->execute([$address]);
if ((int) $cnt->fetchColumn() >= 50) {
    Security::json(['status' => 'ok', 'message' => 'Report noted. This site already has many pending reports.']);
}

$db->prepare(
    "INSERT INTO reports (address, reason, detail, ip_hash) VALUES (?, ?, ?, ?)"
)->execute([$address, $reason, $detail, $ipHash]);

Security::json(['status' => 'ok', 'message' => 'Report submitted. Thank you.']);
