<?php
/**
 * POST /api/search
 * Body: {"q": "keyword", "page": 1}
 * Public (rate-limited), no auth needed.
 */

require_once __DIR__ . '/../includes/security.php';

Security::requirePost();

$cfg = require __DIR__ . '/../includes/config.php';

// Rate limit
if (!Security::checkRate(Security::clientIp(), $cfg['rate_limit'], $cfg['rate_window_sec'])) {
    Security::json(['error' => 'Rate limit exceeded'], 429);
}

// Parse input
$input = json_decode(file_get_contents('php://input'), true) ?: $_POST;
$query = trim($input['q'] ?? '');
$page  = max(1, intval($input['page'] ?? 1));
$limit = $cfg['results_per_page'];
$offset = ($page - 1) * $limit;

if ($query === '' || mb_strlen($query) > $cfg['max_query_length']) {
    Security::json(['error' => 'Invalid query', 'results' => [], 'total' => 0]);
}

// Fix #8: Block unsafe search terms (same filter as frontend)
$_safeExact = ['cp','xxx','fap'];
$_safeBoundary = ['child','pedo','pthc','loli','shota','jailbait','underage','preteen',
    'infant','toddler','molest','rape','incest','porn','sex','nude','naked',
    'hentai','nsfw','bdsm','fetish','escort','webcam','camgirl','onlyfans',
    'milf','teen','nymphet','kiddy','csam'];
$_safeWords = preg_split('/\s+/', strtolower(trim($query)));
foreach ($_safeWords as $_sw) {
    if (in_array($_sw, $_safeExact) || in_array($_sw, $_safeBoundary)) {
        Security::json(['error' => 'Query blocked', 'results' => [], 'total' => 0]);
    }
}

$db = DB::get();

// Sanitize for FULLTEXT: strip special chars that break BOOLEAN MODE
$ftQuery = preg_replace('/[^\p{L}\p{N}\s.\-]/u', '', $query);
$ftQuery = trim($ftQuery);

if ($ftQuery === '') {
    Security::json(['results' => [], 'total' => 0, 'query' => $query]);
}

// Build boolean mode terms: each word becomes +word*
$words = preg_split('/\s+/', $ftQuery);
$boolTerms = implode(' ', array_map(fn($w) => '+' . $w . '*', $words));

// Count total matches
$countSql = "SELECT COUNT(*) as cnt FROM onions
             WHERE MATCH(title, meta_description, meta_keywords, tech_stack, tags, notes, cti_name, cti_type, server_software)
             AGAINST(? IN BOOLEAN MODE)
             AND status != 'unknown'";
$stmt = $db->prepare($countSql);
$stmt->execute([$boolTerms]);
$total = (int) $stmt->fetchColumn();

// Fetch page of results, scored by relevance
$sql = "SELECT
            address, title, status, cti_type, cti_name,
            meta_description, tech_stack, server_software,
            page_language, tags, page_size, response_time_ms,
            link_count, form_count, image_count,
            hits, offline_streak, last_seen, first_seen,
            thumbnail_path,
            MATCH(title, meta_description, meta_keywords, tech_stack, tags, notes, cti_name, cti_type, server_software)
                AGAINST(? IN BOOLEAN MODE) AS relevance
        FROM onions
        WHERE MATCH(title, meta_description, meta_keywords, tech_stack, tags, notes, cti_name, cti_type, server_software)
              AGAINST(? IN BOOLEAN MODE)
              AND status != 'unknown'
        ORDER BY
            CASE status WHEN 'online' THEN 0 ELSE 1 END,
            relevance DESC,
            hits DESC
        LIMIT ? OFFSET ?";

$stmt = $db->prepare($sql);
$stmt->execute([$boolTerms, $boolTerms, $limit, $offset]);
$results = $stmt->fetchAll();

// Get page counts for each result
$addresses = array_column($results, 'address');
$pageCounts = [];
if ($addresses) {
    $placeholders = implode(',', array_fill(0, count($addresses), '?'));
    $pcStmt = $db->prepare(
        "SELECT o.address, COUNT(p.id) as page_count
         FROM onions o LEFT JOIN pages p ON p.onion_id = o.id
         WHERE o.address IN ($placeholders)
         GROUP BY o.address"
    );
    $pcStmt->execute($addresses);
    foreach ($pcStmt->fetchAll() as $r) {
        $pageCounts[$r['address']] = (int) $r['page_count'];
    }
}

// Format output
$out = [];
foreach ($results as $r) {
    $out[] = [
        'address'     => $r['address'],
        'title'       => $r['title'],
        'status'      => $r['status'],
        'cti_type'    => $r['cti_type'],
        'cti_name'    => $r['cti_name'],
        'description' => mb_substr($r['meta_description'], 0, 300),
        'tech'        => $r['tech_stack'],
        'server'      => $r['server_software'],
        'language'    => $r['page_language'],
        'tags'        => json_decode($r['tags'] ?: '[]'),
        'pages'       => $pageCounts[$r['address']] ?? 0,
        'size'        => (int) $r['page_size'],
        'speed_ms'    => (int) $r['response_time_ms'],
        'links'       => (int) $r['link_count'],
        'last_seen'   => $r['last_seen'],
        'first_seen'  => $r['first_seen'],
        'has_thumb'   => $r['thumbnail_path'] !== '',
    ];
}

// Log anonymized query
try {
    $db->prepare("INSERT INTO search_log (query_hash, result_count) VALUES (?, ?)")
       ->execute([hash('sha256', $query), $total]);
} catch (\Throwable $e) { /* silent */ }

Security::json([
    'results' => $out,
    'total'   => $total,
    'page'    => $page,
    'pages'   => max(1, ceil($total / $limit)),
    'query'   => $query,
]);
