<?php
/**
 * POST /api/submit.php — Self-service onion submission
 * Rate limited, HMAC-signed math captcha.
 */
require_once __DIR__ . '/../includes/security.php';
Security::requirePost();

// Fix #9: CSRF check
$_submitRaw = json_decode(file_get_contents('php://input'), true) ?: $_POST;
$_sCsrf = $_submitRaw['_csrf'] ?? '';
$_cCsrf = $_COOKIE['_csrf'] ?? '';
if ($_cCsrf === '' || !hash_equals($_cCsrf, $_sCsrf)) {
    Security::json(['error' => 'Invalid request token. Refresh and try again.'], 403);
}

$cfg = require __DIR__ . '/../includes/config.php';

if (!Security::checkRate(Security::clientIp() . ':submit', 3, 3600)) {
    Security::json(['error' => 'Too many submissions. Try later.'], 429);
}

// Captcha secret — derived from first API key hash in DB or fallback
$captchaSecret = 'inumbra-captcha-' . ($cfg['db_name'] ?? 'salt');

$input = $_submitRaw;
$address = strtolower(trim($input['address'] ?? ''));
$desc    = mb_substr(trim($input['description'] ?? ''), 0, 500);
$answer  = intval($input['captcha_answer'] ?? -1);
$token   = trim($input['captcha_token'] ?? '');

// Validate HMAC-signed captcha
// Fix #28: Token format: "ts:hmac(answer:ts)"
$parts = explode(':', $token);
if (count($parts) !== 2) {
    Security::json(['error' => 'Invalid captcha token'], 400);
}
[$ts, $hmac] = $parts;
$ts = intval($ts);

// Check expiry (5 minutes)
if (abs(time() - $ts) > 300) {
    Security::json(['error' => 'Captcha expired. Refresh and try again.'], 400);
}
// Check HMAC — verify the submitted answer matches what was signed
$expected = hash_hmac('sha256', "$answer:$ts", $captchaSecret);
if (!hash_equals($expected, $hmac)) {
    Security::json(['error' => 'Wrong captcha answer'], 400);
}

// Validate address (v3 only)
if (!preg_match('/^[a-z2-7]{56}\.onion$/', $address)) {
    Security::json(['error' => 'Invalid .onion address (v3 only, 56 chars)'], 400);
}

$db = DB::get();

// Check blocklist
$bl = $db->prepare("SELECT 1 FROM blocklist WHERE hash_md5=?");
$bl->execute([md5($address)]);
if ($bl->fetch()) { Security::json(['error' => 'This address is blocked'], 400); }

// Check dupe submission (any status)
$dup = $db->prepare("SELECT status FROM submissions WHERE address=? ORDER BY submitted_at DESC LIMIT 1");
$dup->execute([$address]); $dupRow = $dup->fetch();
if ($dupRow) {
    if ($dupRow['status'] === 'pending') Security::json(['status' => 'already_submitted']);
    if ($dupRow['status'] === 'accepted') Security::json(['status' => 'already_indexed']);
}

// Check already in index
$ex = $db->prepare("SELECT 1 FROM onions WHERE address=?"); $ex->execute([$address]);
if ($ex->fetch()) { Security::json(['status' => 'already_indexed']); }

$db->prepare("INSERT INTO submissions (address,description,ip_hash) VALUES (?,?,?)")
   ->execute([$address, $desc, hash('sha256', Security::clientIp())]);
Security::json(['status' => 'ok', 'message' => 'Submitted! It will be reviewed and crawled.']);
