#!/usr/bin/env php
<?php
/**
 * In Umbra — Setup Script
 * Run: php setup.php
 *
 * 1. Copies config.example.php → config.php (if missing)
 * 2. Tests database connection
 * 3. Generates an API key for the crawler
 */

echo "══════════════════════════════════════\n";
echo "  Umbra — Setup\n";
echo "══════════════════════════════════════\n\n";

$configPath = __DIR__ . '/includes/config.php';
$examplePath = __DIR__ . '/includes/config.example.php';

// Step 1: Config
if (!file_exists($configPath)) {
    if (file_exists($examplePath)) {
        copy($examplePath, $configPath);
        echo "✓ Created includes/config.php from example\n";
        echo "  → Edit it with your database credentials\n\n";
    } else {
        echo "✗ Missing config.example.php\n";
        exit(1);
    }
} else {
    echo "✓ Config already exists\n";
}

$cfg = require $configPath;

// Step 2: Test DB
echo "\nTesting database connection...\n";
try {
    require_once __DIR__ . '/includes/db.php';
    $db = DB::get();
    $db->query("SELECT 1");
    echo "✓ Database connected ({$cfg['db_name']})\n";

    // Check if tables exist
    $tables = $db->query("SHOW TABLES")->fetchAll(PDO::FETCH_COLUMN);
    $needed = ['onions', 'pages', 'blocklist', 'api_keys', 'rate_limits', 'search_log',
               'reports', 'uptime_log', 'trending_terms', 'site_changes', 'canaries',
               'submissions', 'snapshots', 'removal_requests'];
    $missing = array_diff($needed, $tables);
    if ($missing) {
        echo "  → Missing tables: " . implode(', ', $missing) . "\n";
        echo "  → Run: mysql -u {$cfg['db_user']} -p {$cfg['db_name']} < sql/schema.sql\n";
    } else {
        echo "✓ All tables present\n";
    }
} catch (\Throwable $e) {
    echo "✗ Database error: {$e->getMessage()}\n";
    echo "  → Check your credentials in includes/config.php\n";
    exit(1);
}

// Step 3: Generate API key
echo "\n";
$existingKeys = $db->query("SELECT COUNT(*) FROM api_keys WHERE active=1")->fetchColumn();
if ($existingKeys > 0) {
    echo "ℹ {$existingKeys} active API key(s) already exist\n";
    echo "  Generate a new one? [y/N]: ";
    $answer = trim(fgets(STDIN));
    if (strtolower($answer) !== 'y') {
        echo "\nSetup complete.\n";
        exit(0);
    }
}

$rawKey = bin2hex(random_bytes(32)); // 64-char hex key
$keyHash = hash('sha256', $rawKey);
$label = 'crawler-' . date('Ymd-His');

$db->prepare("INSERT INTO api_keys (key_hash, label, permissions) VALUES (?, ?, 'write')")
   ->execute([$keyHash, $label]);

echo "\n╔══════════════════════════════════════════════════════════════════╗\n";
echo "║  NEW API KEY (save this — it cannot be retrieved later):       ║\n";
echo "╠══════════════════════════════════════════════════════════════════╣\n";
echo "║  {$rawKey}  ║\n";
echo "╠══════════════════════════════════════════════════════════════════╣\n";
echo "║  Label: {$label}                                  ║\n";
echo "╚══════════════════════════════════════════════════════════════════╝\n";
echo "\nUse this in your crawler config:\n";
echo "  API_KEY = \"{$rawKey}\"\n";
echo "  API_URL = \"https://your-domain.com/api/ingest.php\"\n";

echo "\n✓ Setup complete.\n";
